from __future__ import absolute_import

from sage.all import *
from .mixed_monomial import MixedMonomial


############################################################################
############################################################################
# MixedPolynomial
############################################################################
############################################################################
class MixedPolynomial(RingElement):

    def __init__(self,parent,f,*args):        
        RingElement.__init__(self, parent)
        
        if args:
            self.__mons = args[0]
            self.__coeffs = f   
        
        else:
            d = f.monomial_coefficients()
            d = [(d[key],MixedMonomial(parent,key)) for key in d]
            d.sort(key=lambda p: p[1])
            self.__mons = [m for c,m in d]
            self.__coeffs = [c for c,m in d]
        
############################################################################
    def lm(self): return self.__mons[-1]
    def monomials(self): return self.__mons
    def coefficients(self): return self.__coeffs
    def degree(self): return len(self.__mons[-1])
############################################################################
    def __copy__(self):
        mons = [copy(m) for m in self.__mons]
        coeffs = [copy(c) for c in self.__coeffs]
        return MixedPolynomial(self.parent(), coeffs, mons)
############################################################################
    def __repr__(self):
         P = self.parent()
         F = FreeAlgebra(P.base_ring(), P.gens())
         f = '+'.join(str(c) + "*" + str(m).replace('.','*') for c,m in zip(self.__coeffs, self.__mons))
         if f:
             return str(F(f))
         else: 
             return '0'
###########################################################################
    def __eq__(self, other):
        if len(self.__coeffs) != len(other.__coeffs): return False
        return self.__coeffs == other.__coeffs and self.__mons == other.__mons
    def __ne__(self,other):
        return not (self == other)        
############################################################################
    def __hash__(self):
        return hash((tuple(self.__coeffs),tuple(self.__mons)))
############################################################################
    def __mul__(self, other):
        f = copy(self)
        if other in self.parent().base_ring():
            f.__coeffs = [other * c for c in f.__coeffs]
        else:
            f.__mons = [m * other for m in f.__mons]
        return f
############################################################################
    def make_monic(self):
        """
        Really update self
        """
        lc = self.__coeffs[-1]
        if lc != 1: self.__coeffs = [c / lc for c in self.__coeffs]
        return lc
############################################################################
    def lrmul(self, l, r):      
        f = copy(self)
        f.__mons = [m.lrmul(l,r) for m in f.__mons]
        return f
############################################################################
    def lmul(self, l):      
        f = copy(self)
        f.__mons = [m.lmul(l) for m in f.__mons]
        return f
############################################################################
    def rmul(self, r):      
        f = copy(self)
        f.__mons = [m.rmul(r) for m in f.__mons]
        return f
# ############################################################################
#     cpdef to_normal(NCPoly self, Parent=None):
#         def str s
#         def Rational c
#         def NCMonomial m
#         
#         if not Parent:
#             Parent = FreeAlgebra(QQ,len(global_data.vars_), global_data.vars_)
#         s = ""
#         for c,m in zip(self._coeffs,self._mons):
#              s += "".join([str(c), "*", str(m), "+"])
#         return Parent(s + "0")