
from __future__ import absolute_import

from .auxiliary import flatten

from time import time

############################################################################
############################################################################
# Ambiguities
############################################################################
############################################################################
class Ambiguity:
    """
    Overlap = fiC - Afj
    Inclusion = fi - AfjC
    """
        
    def __init__(self, ABC, A, C, i, j, is_overlap):
        self.__ABC = ABC
        self.__A = A
        self.__C = C
        self.__i = i
        self.__j = j
        self.__is_overlap = is_overlap
############################################################################     
    def ABC(self): return self.__ABC
    def A(self): return  self.__ABC[:self.__A]
    def C(self): return self.__ABC[self.__C:]
    def i(self): return self.__i
    def j(self): return self.__j
    def __len__(self): return len(self.__ABC)
    def is_overlap(self): return self.__is_overlap      
############################################################################     
    def __eq__(self,other):
       return self.__i == other.__i and \
              self.__j == other.__j and \
              self.__A == other.__A and \
              self.__C == other.__C and \
              self.__ABC == other.__ABC and \
              self.__is_overlap == other.__is_overlap
    def __ne__(self,other):
        return not (self == other)
#############################################################################
    def __hash__(self):
       return hash( (self.__ABC,self.__A,self.__C,self.__i,self.__j,self.__is_overlap) )
#############################################################################
    def __repr__(self):
        A = self.A()
        C = self.C()
        if not A: A = '1'
        if not C: C = '1'
        if self.__is_overlap:
            s = "Overlap" if self.__is_overlap == 1 else "External"
            s += "(" + "1 x " + str(A) + ", " + str(C) + " x 1, "
        else:
            s = "Inclusion" + "(" + "1 x 1, " + str(A) + " x " + str(C) + ", "
        
        s += str(self.__i) + ", " + str(self.__j) + ")"
        s += "\n ABC = " + str(self.__ABC)
        return s
############################################################################
    @staticmethod
    def generate(a, b, maxdeg):
        
        i,v = a
        j,w = b
        if v.comm_mon() != 1 and w.comm_mon() != 1:
            g = v.comm_mon().gcd(w.comm_mon())
        else: 
            g = 1
        deg_lcm = v.comm_mon().lcm(w.comm_mon()).degree()
        
        
        v = v.nc_mon()
        w = w.nc_mon()
                
        lv = len(v)
        lw = len(w)
        m = min(lv,lw)

        # generate overlaps
        amb  = [Ambiguity(v + w[k:],len(v[:-k]),lv,i,j,True) for k in range(1,m) if v[-k:] == w[:k]]
        amb += [Ambiguity(w + v[k:],len(w[:-k]),lw,j,i,True) for k in range(1,m) if w[-k:] == v[:k]]
        
        # generate inclusions
        if i != j:
            if len(v) > len(w):
                # in this case v is not the empty word
                # so no worries
                amb += [ Ambiguity(v,k,k+lw,i,j,False) for k in v.factor_occurrences_in(w) ]
            else:
                # here w might be the empty word
                if w:
                    amb += [ Ambiguity(w,k,k+lv,j,i,False) for k in w.factor_occurrences_in(v) ]
            
        amb = [a for a in amb if len(a) + deg_lcm <= maxdeg]
         
        # generate external ambiguities - but only if commutative gcd is nontrivial
        # if gcd = 1, all external ambiguities are trivial syzygies
        if g != 1: 
            W = v.parent()
            D = maxdeg - (lv + lw + deg_lcm)
            for l in range(D):
                amb += flatten([[Ambiguity(v + m + w, lv + len(m), lv, i, j, 2),\
                             Ambiguity(w + m + v, lw + len(m), lw, j, i, 2)]\
                            for m in W.iterate_by_length(l)]) 
              
        return list(set(amb)) 
############################################################################
    def to_crit_pair(self, gi, gj): 
        A = self.A()
        C = self.C()
        
        P = gi.parent().algebra().P()  
        lcm = gi.lm().lcm(gj.lm())
        ci = P(lcm / gi.lm().comm_mon())
        cj = P(lcm / gj.lm().comm_mon())

        if self.__is_overlap:
            g1 = gi.rmul(C) * ci
            g2 = gj.lmul(A) * cj
        else:
            g1 = gi * ci
            g2 = gj.lrmul(A,C) * cj
            
        s1 = g1.sig()
        s2 = g2.sig()
                    
        if s1 == s2:
            g1.set_degree(-2)
            return g1
            
            
        if g1.poly() == g2.poly(): 
            d = -1
        else:
            d = len(self)  
               
        if s1 < s2:
            g2.set_degree(d)
            return g2
        else:
            g1.set_degree(d)
            return g1
            
############################################################################
#     cdef tuple to_crit_pair_poly(Ambiguity self, NCPoly gi, NCPoly gj):
#         cdef str A,C
#         cdef NCPoly g1,g2
#         cdef Py_ssize_t d
#         
#         A,C = self.AC()
# 
#         if not self._is_overlap:
#             g1 = gi
#             g2 = gj.lrmul(A,C)
#         else:
#             g1 = gi.rmul(C)
#             g2 = gj.lmul(A)
#                  
#         d = -1 if g1 == g2 else self._deg  
#        
#         return d,g1,g2
#     
############################################################################
#     cdef bint is_redundant(Ambiguity self, str V, Py_ssize_t k):
#         if self._is_overlap: return self.overlap_test(V)
#         else: return self.incl_test(V,k)
############################################################################        
#     cdef bint overlap_test(Ambiguity self, str V):
#         """
#         a) V | m for some m in {A,B,C}, or
#         b) ABC = LVR and
#             i. |L| = 0 and |R| < |C|, or
#             ii. 0 < |L| < |A|, or
#             iii. |L| >= |A| and |R| > 0
#         """
#         cdef str ABC,A,C
#         cdef Py_ssize_t L,R
#         
#         A,C = self.AC()
#         if V in A or V in C or V in self._ABC[self._A : self._C]:
#             return True
#         ABC = self._ABC
# 
#         L = ABC.find(V)
#         R = len(ABC) - (L + len(V))
#         if L == 0 and R < len(ABC) - self._C: return True
#         if L > 0 and L < self._A: return True
#         if R > 0 and L >= self._A: return True
#         return False
# ############################################################################
#     cdef bint incl_test(Ambiguity self, str V, Py_ssize_t k):
#         """
#         k < j and 
#         a) V | m for some m in {A,C}, or
#         b) V | B and |AC| > 0, or
#         c) B | V and |V| < |ABC|
#         """
#         cdef str A,B,C
#         
#         if k >= self._j: return False
#         
#         A,C = self.AC()
#         if V in A or V in C: return True
#         
#         B = self._ABC[self._A : self._C]
#         
#         if V in B and A and C: return True
#         if B in V and len(V) < self._deg: return True
#         return False